<?php
/**
 * Plugin Name: Woo WhatsApp Quick Chat
 * Description: آیکون واتس‌اپ کنار شماره مشتری در سفارش‌های ووکامرس. تمیزکاری شماره و افزودن 98 در صورت نبودن کد کشور.
 * Version: 1.0.2
 * Author: You
 * License: GPL2
 */

if (!defined('ABSPATH')) exit;

/**
 * نرمال‌سازی شماره:
 * - حذف فاصله و کاراکترهای غیر عددی و +
 * - حالت‌های قابل پشتیبانی:
 *   +98xxxxxxxxxx  -> 98xxxxxxxxxx
 *   0098xxxxxxxxxx -> 98xxxxxxxxxx
 *   0xxxxxxxxxx    -> 98xxxxxxxxx  (صفر اول حذف و 98 اضافه می‌شود)
 *   98xxxxxxxxxx   -> بدون تغییر
 *   xxxxxxxxxx     -> اگر 10 تا 11 رقم و با 9 شروع شود، 98 اضافه می‌شود
 * - خروجی مناسب برای WhatsApp Web: فقط ارقام، بدون +
 */
function wwqc_normalize_phone_for_whatsapp($raw) {
    if (!$raw) return '';

    $trimmed = trim($raw);

    // اجازه می‌دهیم + فقط برای شناسایی محفوظ بماند، بعدا حذفش می‌کنیم
    $has_plus = (substr($trimmed, 0, 1) === '+');

    // حذف همه چیز به جز ارقام
    $digits = preg_replace('/\D+/', '', $trimmed);
    if ($digits === '') return '';

    // اگر با 00 شروع شود -> بین‌المللی
    if (strpos($digits, '00') === 0) {
        $digits = substr($digits, 2); // 0098... -> 98...
    }

    // اگر با 0 شروع شود -> محلی ایران فرض می‌کنیم
    if (strpos($digits, '0') === 0) {
        $digits = '98' . substr($digits, 1);
    }

    // اگر با +98 بود، بالا + حذف شده و الان باید 98... باشد
    // اگر هیچ کد کشوری نداشت و طول شبیه ایران بود، 98 اضافه کن
    // الگوی شماره موبایل ایران معمولا 98 + 9XXXXXXXXX است
    if (strpos($digits, '98') !== 0) {
        // اگر 10 یا 11 رقم و با 9 شروع می‌شود، ایران فرض می‌کنیم
        if (preg_match('/^9\d{9}$/', $digits)) {
            $digits = '98' . $digits;
        } elseif (preg_match('/^09\d{9}$/', $digits)) {
            $digits = '98' . substr($digits, 1);
        }
        // در غیر این صورت دست نمی‌زنیم تا اگر از کشور دیگری بود خراب نشود
    }

    return $digits; // بدون علامت +
}

/** ساخت لینک WhatsApp Web */
function wwqc_whatsapp_web_url($phone_digits_no_plus) {
    if (!$phone_digits_no_plus) return '';
    // اگر متن اولیه خواستی:
    // $text = urlencode('سلام وقتتون بخیر');
    // return "https://web.whatsapp.com/send?phone={$phone_digits_no_plus}&text={$text}";
    return "https://web.whatsapp.com/send?phone={$phone_digits_no_plus}";
}

/** آیکون واتس‌اپ SVG سبک */
function wwqc_whatsapp_svg($size = 18) {
    $size = intval($size);
    return '
    <svg width="'.$size.'" height="'.$size.'" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg" aria-hidden="true" focusable="false" style="vertical-align:middle">
      <path d="M19.11 17.12c-.3-.15-1.77-.87-2.05-.97-.28-.1-.49-.15-.7.15-.21.3-.8.97-.98 1.17-.18.2-.36.22-.66.07-.3-.15-1.27-.47-2.42-1.5-.9-.8-1.5-1.77-1.68-2.07-.17-.3-.02-.47.13-.62.13-.13.3-.36.44-.54.15-.18.2-.3.3-.5.1-.2.05-.37-.02-.52-.07-.15-.7-1.68-.96-2.3-.25-.6-.5-.5-.7-.5h-.6c-.2 0-.52.07-.8.37-.28.3-1.05 1.02-1.05 2.5s1.08 2.9 1.23 3.1c.15.2 2.13 3.25 5.15 4.55.72.31 1.28.49 1.72.63.72.23 1.38.2 1.9.12.58-.09 1.77-.72 2.02-1.42.25-.7.25-1.3.18-1.42-.07-.12-.25-.2-.55-.35z" />
      <path d="M16 3C8.82 3 3 8.82 3 16c0 2.31.62 4.47 1.7 6.34L3 29l6.85-1.78C11.62 28.38 13.73 29 16 29c7.18 0 13-5.82 13-13S23.18 3 16 3zm0 23.6c-2.04 0-3.94-.6-5.53-1.63l-.4-.25-4.07 1.06 1.09-3.97-.26-.41A10.55 10.55 0 0 1 5.4 16C5.4 10.7 9.7 6.4 15 6.4S24.6 10.7 24.6 16 20.3 26.6 16 26.6z"/>
    </svg>';
}

/** ستون در لیست سفارش‌ها */
add_filter('manage_edit-shop_order_columns', function ($columns) {
    $new = [];
    foreach ($columns as $key => $label) {
        $new[$key] = $label;
        if ($key === 'order_total') $new['wwqc_whatsapp'] = 'WhatsApp';
    }
    if (!isset($new['wwqc_whatsapp'])) $new['wwqc_whatsapp'] = 'WhatsApp';
    return $new;
}, 20);

add_action('manage_shop_order_posts_custom_column', function ($column, $post_id) {
    if ($column !== 'wwqc_whatsapp') return;

    $order = wc_get_order($post_id);
    if (!$order) return;

    $phone_raw = $order->get_billing_phone();
    if (!$phone_raw) { echo '<span style="opacity:.6">—</span>'; return; }

    $phone_digits = wwqc_normalize_phone_for_whatsapp($phone_raw);
    if (!$phone_digits) { echo '<span style="opacity:.6">—</span>'; return; }

    $url = esc_url(wwqc_whatsapp_web_url($phone_digits));
    $title = esc_attr('باز کردن WhatsApp Web');
    echo '<a href="'.$url.'" target="_blank" rel="noopener noreferrer" title="'.$title.'" style="display:inline-flex;align-items:center;gap:6px;text-decoration:none">';
    echo wwqc_whatsapp_svg(18);
    echo '<span style="font-size:12px">Chat</span>';
    echo '</a>';
}, 20, 2);

/** دکمه در صفحه ویرایش سفارش */
add_action('woocommerce_admin_order_data_after_billing_address', function ($order) {
    if (!$order instanceof WC_Order) return;

    $phone_raw = $order->get_billing_phone();
    if (!$phone_raw) return;

    $phone_digits = wwqc_normalize_phone_for_whatsapp($phone_raw);
    if (!$phone_digits) return;

    $url = esc_url(wwqc_whatsapp_web_url($phone_digits));
    echo '<p style="margin-top:8px">';
    echo '<a href="'.$url.'" target="_blank" rel="noopener noreferrer" class="button" style="display:inline-flex;align-items:center;gap:8px">';
    echo wwqc_whatsapp_svg(16);
    echo '<span>WhatsApp Web</span>';
    echo '</a>';
    echo '</p>';
}, 20);
