<?php
/**
 * Plugin Name: Woo REST SKU Search Enhancment (Augment Default Search)
 * Description: Augments /wp-json/wc/v3/products?search=... to also include SKU matches (partial or exact) without adding new routes or hacking SQL.
 * Version:     1.0.0
 * Author:      The Hippoo team
 * License:     GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

final class WCRestSkuAugment {
    const MODE = 'partial'; // 'partial' or 'exact'

    // Per-request stash
    private static $active = false;
    private static $term   = '';
    private static $page   = 1;
    private static $per    = 10;

    public static function boot() {
        // Detect the core products collection route before controller runs
        add_filter('rest_request_before_callbacks', [__CLASS__, 'maybe_flag_request'], 10, 3);
        // After controller prepares the response, merge in SKU hits
        add_filter('rest_post_dispatch',            [__CLASS__, 'maybe_augment_response'], 10, 3);
    }

    public static function is_products_collection($request) {
        // wc/v3/products collection only
        $route = $request->get_route();
        if (!is_string($route)) return false;
        // Match /wc/v3/products and /wc/v3/products/ (no IDs)
        return (bool) preg_match('#^/wc/v3/products/?$#', $route);
    }

    public static function maybe_flag_request($response, $handler, $request) {
        self::$active = false;
        self::$term   = '';
        self::$page   = 1;
        self::$per    = 10;

        if (!self::is_products_collection($request)) {
            return $response;
        }
        $q = trim((string) $request->get_param('search'));
        if ($q === '') {
            return $response;
        }
        self::$active = true;
        self::$term   = $q;

        // Track paging so we can be polite with how much we add
        $per = (int) $request->get_param('per_page');
        $page= (int) $request->get_param('page');
        if ($per > 0) self::$per  = min(100, $per);
        if ($page> 0) self::$page = $page;

        return $response;
    }

    public static function maybe_augment_response($result, $server, $request) {
        if (!self::$active || !self::is_products_collection($request)) {
            return $result;
        }
        if (!($result instanceof WP_REST_Response)) {
            return $result;
        }

        $data = $result->get_data();
        if (!is_array($data)) {
            return $result;
        }

        // Build a set of already-returned product IDs
        $existing_ids = [];
        foreach ($data as $row) {
            if (isset($row['id'])) $existing_ids[(int) $row['id']] = true;
        }

        $sku_ids = self::find_product_ids_by_sku(self::$term, self::MODE === 'partial');
        if (!$sku_ids) {
            return $result;
        }

        // Bubble variations to parent products and keep only products
        $parent_ids = [];
        foreach ($sku_ids as $pid) {
            $post = get_post($pid);
            if (!$post) continue;
            if ($post->post_type === 'product_variation') {
                $parent = (int) $post->post_parent;
                if ($parent) $parent_ids[$parent] = true;
            } elseif ($post->post_type === 'product') {
                $parent_ids[$post->ID] = true;
            }
        }
        $parent_ids = array_keys($parent_ids);

        // Drop any IDs already present
        $new_ids = array_values(array_filter($parent_ids, function($id) use ($existing_ids) {
            return empty($existing_ids[$id]);
        }));

        if (!$new_ids) {
            return $result;
        }

        // Prepare responses using the same controller to match schema
        if (!class_exists('WC_REST_Products_Controller')) {
            return $result; // Woo not loaded (shouldn't happen on this route)
        }
        $controller = new WC_REST_Products_Controller();

        // Respect per_page: add up to the remaining slots on this page
        $remaining = max(0, self::$per - count($data));
        if ($remaining === 0) {
            // Nothing to add on this page. Keep headers untouched.
            return $result;
        }
        $new_ids = array_slice($new_ids, 0, $remaining);

        foreach ($new_ids as $pid) {
            $product = wc_get_product($pid);
            if (!$product) continue;

            // Use controller’s schema/formatting
            $prepared = $controller->prepare_object_for_response($product, $request);
            if ($prepared instanceof WP_REST_Response) {
                $data[] = $prepared->get_data();
            }
        }

        // Update totals headers so pagination stays truthful
        // We can only approximate: total = original total + unique additions
        $orig_total = (int) $result->get_headers()['X-WP-Total'] ?? 0;
        $orig_pages = (int) $result->get_headers()['X-WP-TotalPages'] ?? 0;

        $new_total = $orig_total + count($new_ids);
        $new_pages = ($new_total > 0) ? (int) ceil($new_total / max(1, self::$per)) : $orig_pages;

        $result->set_data($data);
        $result->header('X-WP-Total', (string) $new_total);
        $result->header('X-WP-TotalPages', (string) $new_pages);

        return $result;
    }

    /**
     * Return an array of product/variation post IDs whose SKU matches $term.
     * If $partial=false, exact match; else LIKE %term%.
     */
    private static function find_product_ids_by_sku(string $term, bool $partial = true): array {
        global $wpdb;

        if ($term === '') return [];

        $like = '%' . $wpdb->esc_like($term) . '%';

        if ($partial) {
            $sql = $wpdb->prepare("
                SELECT p.ID
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->postmeta} pm
                    ON pm.post_id = p.ID AND pm.meta_key = '_sku'
                WHERE p.post_type IN ('product','product_variation')
                  AND p.post_status IN ('publish','private')  /* mimic WC visibility */
                  AND pm.meta_value LIKE %s
                LIMIT 500
            ", $like);
        } else {
            $sql = $wpdb->prepare("
                SELECT p.ID
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->postmeta} pm
                    ON pm.post_id = p.ID AND pm.meta_key = '_sku'
                WHERE p.post_type IN ('product','product_variation')
                  AND p.post_status IN ('publish','private')
                  AND pm.meta_value = %s
                LIMIT 500
            ", $term);
        }

        $ids = $wpdb->get_col($sql);
        return array_values(array_unique(array_map('intval', $ids)));
    }
}

add_action('rest_api_init', ['WCRestSkuAugment', 'boot']);
